<?php

/*
Plugin Name: Sample Plugin
Plugin URI: http://brizpress.com
Description: Mengilustrasikan cara menyertakan update di plugin Anda untuk Lisensi SEJOLI
Author: BrizPress
Author URI: http://brizpress.com
Version: 1.0
License: GNU General Public License v2.0 or later
License URI: http://www.gnu.org/licenses/gpl-2.0.html
*/

/**
* Silakan untuk rincian lebih lanjut visit https://docs.brizpress.com/docs/sejoli-license/implementasi-updater-untuk-plugin-wordpress
 */

// ini adalah URL ping update/pemeriksa lisensi. Ini harus menjadi URL situs yang menginstal SEJOLI
define( 'SEJOLI_SAMPLE_STORE_URL', 'https://YOUR-MEMBERSHIP.com' ); // Anda harus menggunakan nama KONSTAN Anda sendiri, dan pastikan untuk menggantinya di seluruh file ini

// ini adalah nomor versi plugin saat ini.
define( 'SEJOLI_SAMPLE_VERSION', '1.0'); // Anda harus menggunakan nama KONSTAN Anda sendiri, dan pastikan untuk menggantinya di seluruh file ini dan sesuaikan nomor versi

// nama folder/direktori plugin
define( 'SEJOLI_SAMPLE_ITEM_NAME', 'sejoli-sample-plugin' ); // Anda harus menggunakan nama KONSTAN Anda sendiri, dan pastikan untuk menggantinya di seluruh file ini

// nama halaman pengaturan untuk input lisensi yang akan ditampilkan
define( 'SEJOLI_SAMPLE_PLUGIN_LICENSE_PAGE', 'pluginname-license' );

if ( ! class_exists( 'SEJOLI_Plugin_Updater' ) ) {
	// memuat update
	include dirname( __FILE__ ) . '/SEJOLI_Plugin_Updater.php';
}

/**
 * Inisialisasi pembaru. Terhubung ke `init` untuk bekerja dengan
 * wp_version_check cron job, yang memungkinkan auto-updates.
 */
function sejoli_sl_sample_plugin_updater() {

	// Untuk mendukung pembaruan otomatis, ini perlu dijalankan selama tugas cron wp_version_check untuk pengguna yang memiliki hak istimewa.
	$doing_cron = defined( 'DOING_CRON' ) && DOING_CRON;
	if ( ! current_user_can( 'manage_options' ) && ! $doing_cron ) {
		return;
	}

	// mengambil kode lisensi dari DB
	$license_key = trim( get_option( 'sejoli_sample_license_key' ) );

	// pengaturan updater
	$sejoli_updater = new SEJOLI_Plugin_Updater(
		SEJOLI_SAMPLE_STORE_URL,
		__FILE__,
		array(
			'version' => SEJOLI_SAMPLE_VERSION, // nomor versi saat ini
			'license' => $license_key, // kode lisensi (digunakan get_option di atas untuk mengambil dari DB)
			'item_name' => urlencode( SEJOLI_SAMPLE_ITEM_NAME ),  // ID of the product
			'author'  => 'BrizPress', // nama pemilik plugin
		)
	);

}
add_action( 'init', 'sejoli_sl_sample_plugin_updater' );


/************************************
* kode di bawah ini hanya standar saja
* halaman opsi. Gantikan dengan
* milikmu sendiri.
*************************************/

/**
 * Menambahkan halaman lisensi plugin ke menu admin.
 *
 * @return void
 */
function sejoli_sample_license_menu() {
	add_plugins_page(
		__( 'Plugin License' ),
		__( 'Plugin License' ),
		'manage_options',
		SEJOLI_SAMPLE_PLUGIN_LICENSE_PAGE,
		'sejoli_sample_license_page'
	);
}
add_action( 'admin_menu', 'sejoli_sample_license_menu' );

function sejoli_sample_license_page() {
	add_settings_section(
		'sejoli_sample_license',
		__( 'Plugin License' ),
		'sejoli_sample_license_key_settings_section',
		SEJOLI_SAMPLE_PLUGIN_LICENSE_PAGE
	);
	add_settings_field(
		'sejoli_sample_license_key',
		'<label for="sejoli_sample_license_key">' . __( 'License Key' ) . '</label>',
		'sejoli_sample_license_key_settings_field',
		SEJOLI_SAMPLE_PLUGIN_LICENSE_PAGE,
		'sejoli_sample_license',
	);
	?>
	<div class="wrap">
		<h2><?php esc_html_e( 'Plugin License Options' ); ?></h2>
		<form method="post" action="options.php">

			<?php
			do_settings_sections( SEJOLI_SAMPLE_PLUGIN_LICENSE_PAGE );
			settings_fields( 'sejoli_sample_license' );
			submit_button();
			?>

		</form>
	<?php
}

/**
 * Menambahkan konten ke bagian pengaturan.
 *
 * @return void
 */
function sejoli_sample_license_key_settings_section() {
	esc_html_e( 'This is where you enter your license key.' );
}

/**
 * Menampilkan bidang pengaturan kunci lisensi.
 *
 * @return void
 */
function sejoli_sample_license_key_settings_field() {
	$license = get_option( 'sejoli_sample_license_key' );
	$status  = get_option( 'sejoli_sample_license_status' );

	?>
	<p class="description"><?php esc_html_e( 'Enter your license key.' ); ?></p>
	<?php
	printf(
		'<input type="text" class="regular-text" id="sejoli_sample_license_key" name="sejoli_sample_license_key" value="%s" />',
		esc_attr( $license )
	);
	$button = array(
		'name'  => 'sejoli_license_deactivate',
		'label' => __( 'Deactivate License' ),
	);
	if ( 'valid' !== $status ) {
		$button = array(
			'name'  => 'sejoli_license_activate',
			'label' => __( 'Activate License' ),
		);
	}
	wp_nonce_field( 'sejoli_sample_nonce', 'sejoli_sample_nonce' );
	?>
	<input type="submit" class="button-secondary" name="<?php echo esc_attr( $button['name'] ); ?>" value="<?php echo esc_attr( $button['label'] ); ?>"/>
	<?php
}

/**
 * Mendaftarkan pengaturan kunci lisensi di tabel opsi.
 *
 * @return void
 */
function sejoli_sample_register_option() {
	register_setting( 'sejoli_sample_license', 'sejoli_sample_license_key', 'sejoli_sanitize_license' );
}
add_action( 'admin_init', 'sejoli_sample_register_option' );

/**
 * Membersihkan kunci lisensi.
 *
 * @param string  $kode lisensi baru.
 * @return string
 */
function sejoli_sanitize_license( $new ) {
	$old = get_option( 'sejoli_sample_license_key' );
	if ( $old && $old !== $new ) {
		delete_option( 'sejoli_sample_license_status' ); // izin baru sudah masuk, jadi harus aktif kembali
	}

	return sanitize_text_field( $new );
}

/**
 * Mengaktifkan kode lisensi.
 *
 * @return void
 */
function sejoli_sample_activate_license() {

	// dengarkan tombol aktifkan diklik
	if ( ! isset( $_POST['sejoli_license_activate'] ) ) {
		return;
	}

	// menjalankan pemeriksaan keamanan cepat
	if ( ! check_admin_referer( 'sejoli_sample_nonce', 'sejoli_sample_nonce' ) ) {
		return; // keluar jika kita tidak mengklik tombol Activate
	}

	// mengambil lisensi dari database
	$license = trim( get_option( 'sejoli_sample_license_key' ) );
	if ( ! $license ) {
		$license = ! empty( $_POST['sejoli_sample_license_key'] ) ? sanitize_text_field( $_POST['sejoli_sample_license_key'] ) : '';
	}
	if ( ! $license ) {
		return;
	}

	// data untuk dikirim dalam permintaan API
	$api_params = array(
		'sejoli_action'  => 'activate_license',
		'license'     => $license,
		'item_name'   => rawurlencode( SEJOLI_SAMPLE_ITEM_NAME ), // nama folder/direktori plugin
		'url'         => home_url(),
	);

	// Panggil API
	$response = wp_remote_post(
		SEJOLI_SAMPLE_STORE_URL,
		array(
			'timeout'   => 15,
			'sslverify' => false,
			'body'      => $api_params,
		)
	);

	// pastikan responnya kembali oke
	if ( is_wp_error( $response ) || 200 !== wp_remote_retrieve_response_code( $response ) ) {

		if ( is_wp_error( $response ) ) {
			$message = $response->get_error_message();
		} else {
			$message = __( 'An error occurred, please try again.' );
		}
	} else {

		$license_data = json_decode( wp_remote_retrieve_body( $response ) );

		if ( false === $license_data->success ) {

			switch ( $license_data->error ) {

				case 'expired':
					$message = sprintf(
						/* penerjemah: tanggal kedaluwarsa kode lisensi */
						__( 'Kode lisensi Anda telah kadaluarsa pada %s.', 'sejoli-sample-plugin' ),
						date_i18n( get_option( 'date_format' ), strtotime( $license_data->expires, current_time( 'timestamp' ) ) )
					);
					break;

				case 'disabled':
				case 'revoked':
					$message = __( 'Kode lisensi Anda telah dinonaktifkan dan tidak dapat dipergunakan lagi.', 'sejoli-sample-plugin' );
					break;

				case 'missing':
					$message = __( 'Lisensi tidak valid.', 'sejoli-sample-plugin' );
					break;

				case 'invalid' :
					if ( !empty($license_data->error) ) {
						$message = $license_data->error;
					}
					else {
						$message = __( 'Lisensi belum atau tidak bisa diaktifkan. Silahkan coba lagi', 'sejoli-memberareacustomizer' );
					}
					break;
				case 'site_inactive':
					$message = __( 'Lisensi Anda sedang tidak aktif di website ini.', 'sejoli-sample-plugin' );
					break;

				case 'item_name_mismatch':
					/* translators: the plugin name */
					$message = sprintf( __( 'Kode lisensi tidak valid untuk %s, silahkan ganti dengan kode lisensi yang benar.', 'sejoli-sample-plugin' ), SEJOLI_SAMPLE_ITEM_NAME );
					break;

				case 'no_activations_left':
					$message = __( 'Kode lisensi Anda telah mencapai batas limit aktivasi lisensi.', 'sejoli-sample-plugin' );
					break;

				default:
					$message = __( 'An error occurred, please try again.', 'sejoli-sample-plugin' );
					break;
			}
		}
	}

	// Periksa apakah ada pesan yang diteruskan yang merupakan kegagalan
	if ( ! empty( $message ) ) {
		$redirect = add_query_arg(
			array(
				'page'          => SEJOLI_SAMPLE_PLUGIN_LICENSE_PAGE,
				'sl_activation' => 'false',
				'message'       => rawurlencode( $message ),
			),
			admin_url( 'plugins.php' )
		);

		wp_safe_redirect( $redirect );
		exit();
	}

	// $license_data->license akan menjadi "valid" atau "invalid"
	if ( 'valid' === $license_data->license ) {
		update_option( 'sejoli_sample_license_key', $license );
	}
	update_option( 'sejoli_sample_license_status', $license_data->license );
	wp_safe_redirect( admin_url( 'plugins.php?page=' . SEJOLI_SAMPLE_PLUGIN_LICENSE_PAGE ) );
	exit();
}
add_action( 'admin_init', 'sejoli_sample_activate_license' );

/**
*  Menonaktifkan kode lisensi.
 * Ini akan mengurangi jumlah kouta lisensi.
 *
 * @return void
 */
function sejoli_sample_deactivate_license() {

	// dengarkan tombol aktifkan diklik
	if ( isset( $_POST['sejoli_license_deactivate'] ) ) {

		// menjalankan pemeriksaan keamanan cepat
		if ( ! check_admin_referer( 'sejoli_sample_nonce', 'sejoli_sample_nonce' ) ) {
			return; // keluar jika kita tidak mengklik tombol Activate
		}

		// mengambil lisensi dari database
		$license = trim( get_option( 'sejoli_sample_license_key' ) );

		// data untuk dikirim dalam permintaan API kami
		$api_params = array(
			'sejoli_action'  => 'deactivate_license',
			'license'     => $license,
			'item_name'   => rawurlencode( SEJOLI_SAMPLE_ITEM_NAME ), // nama folder/direktori plugin
			'url'         => home_url(),
		);

		// Panggil API
		$response = wp_remote_post(
			SEJOLI_SAMPLE_STORE_URL,
			array(
				'timeout'   => 15,
				'sslverify' => false,
				'body'      => $api_params,
			)
		);

		// pastikan responnya kembali oke
		if ( is_wp_error( $response ) || 200 !== wp_remote_retrieve_response_code( $response ) ) {

			if ( is_wp_error( $response ) ) {
				$message = $response->get_error_message();
			} else {
				$message = __( 'An error occurred, please try again.' );
			}

			$redirect = add_query_arg(
				array(
					'page'          => SEJOLI_SAMPLE_PLUGIN_LICENSE_PAGE,
					'sl_activation' => 'false',
					'message'       => rawurlencode( $message ),
				),
				admin_url( 'plugins.php' )
			);

			wp_safe_redirect( $redirect );
			exit();
		}

		// decode kode data lisensi
		$license_data = json_decode( wp_remote_retrieve_body( $response ) );

		// $license_data->license akan "deactivated" atau "failed"
		if ( 'deactivated' === $license_data->license ) {
			delete_option( 'sejoli_sample_license_status' );
		}

		wp_safe_redirect( admin_url( 'plugins.php?page=' . SEJOLI_SAMPLE_PLUGIN_LICENSE_PAGE ) );
		exit();

	}
}
add_action( 'admin_init', 'sejoli_sample_deactivate_license' );

/**
*  Memeriksa apakah kode lisensi masih valid.
 * Pembaru melakukan ini untuk Anda, jadi ini hanya diperlukan jika Anda mau
 * untuk melakukan sesuatu yang khusus.
 *
 * @return void
 */
function sejoli_sample_check_license() {

	$license = trim( get_option( 'sejoli_sample_license_key' ) );

	$api_params = array(
		'sejoli_action'  => 'check_license',
		'license'     => $license,
		'item_id'     => SEJOLI_SAMPLE_ITEM_ID,
		'item_name'   => rawurlencode( SEJOLI_SAMPLE_ITEM_NAME ),
		'url'         => home_url(),
	);

	// Call the custom API.
	$response = wp_remote_post(
		SEJOLI_SAMPLE_STORE_URL,
		array(
			'timeout'   => 15,
			'sslverify' => false,
			'body'      => $api_params,
		)
	);

	if ( is_wp_error( $response ) ) {
		return false;
	}

	$license_data = json_decode( wp_remote_retrieve_body( $response ) );

	if ( 'valid' === $license_data->license ) {
		echo 'valid';
		exit;
		// lisensi ini masih valid
	} else {
		echo 'invalid';
		exit;
		// lisensi ini tidak berlaku lagi / invalid
	}
}

/**
 * Ini merupakan sarana untuk menangkap kesalahan dari cara aktivasi di atas dan menampilkannya kepada pelanggan
 */
function sejoli_sample_admin_notices() {
	if ( isset( $_GET['sl_activation'] ) && ! empty( $_GET['message'] ) ) {

		switch ( $_GET['sl_activation'] ) {

			case 'false':
				$message = urldecode( $_GET['message'] );
				?>
				<div class="error">
					<p><?php echo wp_kses_post( $message ); ?></p>
				</div>
				<?php
				break;

			case 'true':
			default:
				// Pengembang dapat menempatkan pesan sukses khusus di sini ketika aktivasi berhasil jika memungkinkan.
				break;

		}
	}
}
add_action( 'admin_notices', 'sejoli_sample_admin_notices' );
